# -*- coding: utf-8 -*-
import sys
import json
import re
import time
import urllib.parse
import urllib.request
import socket
import xml.etree.ElementTree as ET

import xbmc
import xbmcaddon
import xbmcgui
import xbmcplugin
import xbmcvfs

ADDON = xbmcaddon.Addon()
HANDLE = int(sys.argv[1])
PLUGIN_URL = sys.argv[0]

DEFAULT_BASE = "https://wagasworld.com/m3u8.php?apikey="
TVWORLD_URL = "https://raw.githubusercontent.com/Free-TV/IPTV/master/playlist.m3u8"

CATEGORY_PRETTY = {
    "Alle": "Alle Streams",
    "- Alle -": "Alle Streams",
    "WiP": "WiP",
    "Movies": "Filme",
    "Cartoons": "Cartoons",
    "Kids": "Kinder",
    "Anime": "Anime",
    "Sitcoms/Familie": "Sitcoms & Familie",
    "SciFi": "Sci‑Fi",
    "FantasyMystery": "Fantasy & Mystery",
    "Crime": "Crime",
    "Med/Drama": "Drama & Medical",
    "Doku": "Doku",
    "Audio": "Audio",
}

CATEGORIES_ICON_DIR = "special://home/addons/plugin.video.wagasworld/resources/media/categories/"
FANART_DIR = "special://home/addons/plugin.video.wagasworld/resources/media/fanart/"

CATEGORY_ICON = {
    "Alle": CATEGORIES_ICON_DIR + "alle.png",
    "- Alle -": CATEGORIES_ICON_DIR + "alle.png",
    "WiP": CATEGORIES_ICON_DIR + "wip.png",
    "Movies": CATEGORIES_ICON_DIR + "movies.png",
    "Cartoons": CATEGORIES_ICON_DIR + "cartoons.png",
    "Kids": CATEGORIES_ICON_DIR + "kids.png",
    "Anime": CATEGORIES_ICON_DIR + "anime.png",
    "Sitcoms/Familie": CATEGORIES_ICON_DIR + "sitcoms.png",
    "SciFi": CATEGORIES_ICON_DIR + "scifi.png",
    "FantasyMystery": CATEGORIES_ICON_DIR + "fantasy.png",
    "Crime": CATEGORIES_ICON_DIR + "crime.png",
    "Med/Drama": CATEGORIES_ICON_DIR + "drama.png",
    "Doku": CATEGORIES_ICON_DIR + "doku.png",
    "Audio": CATEGORIES_ICON_DIR + "audio.png",
}

SOURCE_ICON = {
    "wagas": CATEGORIES_ICON_DIR + "wagas.png",
    "tvworld": CATEGORIES_ICON_DIR + "tvworld.png",
    "favs": CATEGORIES_ICON_DIR + "favs.png",
    "recent": CATEGORIES_ICON_DIR + "recent.png",
}

SOURCE_FANART = {
    "wagas": FANART_DIR + "root_wagas.jpg",
    "tvworld": FANART_DIR + "root_tvworld.jpg",
    "favs": FANART_DIR + "root_favs.jpg",
    "recent": FANART_DIR + "root_recent.jpg",
}

CATEGORY_FANART = {
    "Alle": FANART_DIR + "cat_alle.jpg",
    "- Alle -": FANART_DIR + "cat_alle.jpg",
    "WiP": FANART_DIR + "cat_wip.jpg",
    "Movies": FANART_DIR + "cat_movies.jpg",
    "Cartoons": FANART_DIR + "cat_cartoons.jpg",
    "Kids": FANART_DIR + "cat_kids.jpg",
    "Anime": FANART_DIR + "cat_anime.jpg",
    "Sitcoms/Familie": FANART_DIR + "cat_sitcoms.jpg",
    "SciFi": FANART_DIR + "cat_scifi.jpg",
    "FantasyMystery": FANART_DIR + "cat_fantasy.jpg",
    "Crime": FANART_DIR + "cat_crime.jpg",
    "Med/Drama": FANART_DIR + "cat_drama.jpg",
    "Doku": FANART_DIR + "cat_doku.jpg",
    "Audio": FANART_DIR + "cat_audio.jpg",
}

DEFAULT_FANART = FANART_DIR + "default.jpg"


def log(msg, level=xbmc.LOGINFO):
    xbmc.log("[WAGASWORLD] " + str(msg), level)


def notify(title, message, ms=3500):
    xbmcgui.Dialog().notification(title, message, time=ms)


def build_plugin_url(**query):
    return PLUGIN_URL + "?" + urllib.parse.urlencode(query)


def get_setting(key, default=""):
    try:
        val = ADDON.getSetting(key)
        return val if val is not None and val != "" else default
    except Exception:
        return default


def set_setting(key, value):
    try:
        ADDON.setSetting(key, value)
    except Exception:
        pass


def as_int(s, default=0):
    try:
        return int(str(s).strip())
    except Exception:
        return default


# ---------- Paths ----------
def profile_path():
    return xbmcvfs.translatePath(ADDON.getAddonInfo("profile"))


def ensure_profile_dir():
    p = profile_path()
    if not xbmcvfs.exists(p):
        xbmcvfs.mkdirs(p)


def path_join(a, b):
    return a.rstrip("/\\") + "/" + b.lstrip("/\\")


def fav_file():
    return path_join(profile_path(), "favorites.json")


def recent_file():
    return path_join(profile_path(), "recent.json")


def cache_file(source):
    return path_join(profile_path(), "cache_%s.json" % source)


# ---------- Cache ----------
def cache_ttl_seconds():
    return max(5, as_int(get_setting("cache_ttl", "90"), 90))


def load_cache(source):
    ensure_profile_dir()
    p = cache_file(source)
    if not xbmcvfs.exists(p):
        return None
    try:
        f = xbmcvfs.File(p)
        raw = f.read()
        f.close()
        obj = json.loads(raw) if raw else None
        return obj if isinstance(obj, dict) else None
    except Exception as e:
        log("Cache lesen fehlgeschlagen (%s): %s" % (source, e), xbmc.LOGERROR)
        return None


def save_cache(source, text):
    ensure_profile_dir()
    p = cache_file(source)
    try:
        f = xbmcvfs.File(p, "w")
        f.write(json.dumps({"ts": int(time.time()), "text": text}, ensure_ascii=False))
        f.close()
        return True
    except Exception as e:
        log("Cache speichern fehlgeschlagen (%s): %s" % (source, e), xbmc.LOGERROR)
        return False


def http_get(url, timeout=12):
    log("GET %s" % url)
    headers = {"User-Agent": "Kodi Wagasworld Addon/1.1.0", "Accept": "*/*"}
    req = urllib.request.Request(url, headers=headers)
    with urllib.request.urlopen(req, timeout=timeout) as resp:
        data = resp.read()
        try:
            charset = resp.headers.get_content_charset()
        except Exception:
            charset = None
        return data.decode(charset or "utf-8", errors="replace")


def get_cached_text(source, url):
    ttl = cache_ttl_seconds()
    obj = load_cache(source)
    if obj and "ts" in obj and "text" in obj:
        age = int(time.time()) - int(obj.get("ts", 0))
        if age <= ttl:
            return obj["text"], True
    text = http_get(url)
    save_cache(source, text)
    return text, False


# ---------- Favorites (folders + rename + ordering) ----------
def load_favs():
    ensure_profile_dir()
    path = fav_file()
    if not xbmcvfs.exists(path):
        return []
    try:
        f = xbmcvfs.File(path)
        raw = f.read()
        f.close()
        data = json.loads(raw) if raw else []
        return data if isinstance(data, list) else []
    except Exception as e:
        log("Favs laden fehlgeschlagen: %s" % e, xbmc.LOGERROR)
        return []


def save_favs(favs):
    ensure_profile_dir()
    path = fav_file()
    try:
        f = xbmcvfs.File(path, "w")
        f.write(json.dumps(favs, ensure_ascii=False, indent=2))
        f.close()
        return True
    except Exception as e:
        log("Favs speichern fehlgeschlagen: %s" % e, xbmc.LOGERROR)
        return False


def fav_index_by_url(url):
    url = (url or "").strip()
    if not url:
        return -1
    favs = load_favs()
    for i, f in enumerate(favs):
        if (f.get("url") or "").strip() == url:
            return i
    return -1


def is_fav(url):
    return fav_index_by_url(url) >= 0


def existing_folders():
    favs = load_favs()
    folders = []
    for f in favs:
        name = (f.get("folder") or "").strip()
        if name and name not in folders:
            folders.append(name)
    return folders


def pick_folder(default_name=""):
    folders = existing_folders()
    items = []
    if folders:
        items.extend(folders)
    items.append("➕ Neuer Ordner …")
    preselect = 0
    if default_name and default_name in folders:
        preselect = folders.index(default_name)
    idx = xbmcgui.Dialog().select("Favoriten-Ordner wählen", items, preselect=preselect)
    if idx < 0:
        return ""
    if items[idx] == "➕ Neuer Ordner …":
        name = xbmcgui.Dialog().input("Neuer Ordnername", type=xbmcgui.INPUT_ALPHANUM)
        return (name or "").strip()
    return (items[idx] or "").strip()


def add_fav(it, folder=""):
    url = (it.get("url") or "").strip()
    if not url:
        return False

    favs = load_favs()
    for f in favs:
        if (f.get("url") or "").strip() == url:
            if folder:
                f["folder"] = folder
                save_favs(favs)
            return True

    favs.append({
        "title": it.get("title") or "Stream",
        "url": it.get("url") or "",
        "logo": it.get("logo") or "",
        "group": it.get("group") or "Alle",
        "source": it.get("source") or "wagas",
        "folder": folder or (it.get("folder") or "Meine Favoriten"),
        "added_ts": int(time.time()),
        "last_played_ts": 0
    })
    return save_favs(favs)


def remove_fav(url):
    url = (url or "").strip()
    if not url:
        return False
    favs = load_favs()
    favs2 = [f for f in favs if (f.get("url") or "").strip() != url]
    return save_favs(favs2)


def fav_rename(url):
    idx = fav_index_by_url(url)
    if idx < 0:
        return False
    favs = load_favs()
    cur = favs[idx].get("title") or ""
    name = xbmcgui.Dialog().input("Favorit umbenennen", defaultt=cur, type=xbmcgui.INPUT_ALPHANUM)
    name = (name or "").strip()
    if not name:
        return False
    favs[idx]["title"] = name
    return save_favs(favs)


def fav_move_folder(url):
    idx = fav_index_by_url(url)
    if idx < 0:
        return False
    favs = load_favs()
    cur = (favs[idx].get("folder") or "").strip()
    folder = pick_folder(default_name=cur)
    if not folder:
        return False
    favs[idx]["folder"] = folder
    return save_favs(favs)


def fav_move(url, direction):
    idx = fav_index_by_url(url)
    if idx < 0:
        return False
    favs = load_favs()
    new_idx = idx + direction
    if new_idx < 0 or new_idx >= len(favs):
        return False
    favs[idx], favs[new_idx] = favs[new_idx], favs[idx]
    return save_favs(favs)


def fav_sort(mode):
    favs = load_favs()
    if not favs:
        return False
    if mode == "alpha":
        favs.sort(key=lambda x: (x.get("folder", ""), (x.get("title") or "").lower()))
    elif mode == "recent":
        favs.sort(key=lambda x: int(x.get("last_played_ts") or 0), reverse=True)
    elif mode == "added":
        favs.sort(key=lambda x: int(x.get("added_ts") or 0), reverse=True)
    else:
        return False
    return save_favs(favs)


def update_fav_last_played(url):
    idx = fav_index_by_url(url)
    if idx < 0:
        return
    favs = load_favs()
    favs[idx]["last_played_ts"] = int(time.time())
    save_favs(favs)


# ---------- Recent ----------
def load_recent():
    ensure_profile_dir()
    p = recent_file()
    if not xbmcvfs.exists(p):
        return []
    try:
        f = xbmcvfs.File(p)
        raw = f.read()
        f.close()
        data = json.loads(raw) if raw else []
        return data if isinstance(data, list) else []
    except Exception:
        return []


def save_recent(lst):
    ensure_profile_dir()
    p = recent_file()
    try:
        f = xbmcvfs.File(p, "w")
        f.write(json.dumps(lst, ensure_ascii=False, indent=2))
        f.close()
        return True
    except Exception:
        return False


def add_recent(entry, limit=20):
    url = (entry.get("url") or "").strip()
    if not url:
        return
    lst = load_recent()
    lst = [x for x in lst if (x.get("url") or "").strip() != url]
    entry["ts"] = int(time.time())
    lst.insert(0, entry)
    lst = lst[:limit]
    save_recent(lst)


# ---------- API Key ----------
def ask_for_apikey():
    kb = xbmc.Keyboard("", "API-Key eingeben")
    kb.doModal()
    if kb.isConfirmed():
        apikey = (kb.getText() or "").strip()
        if apikey:
            set_setting("apikey", apikey)
            return apikey
    return ""


def get_apikey():
    apikey = get_setting("apikey", "").strip()
    return apikey if apikey else ask_for_apikey()


def replace_apikey_in_url(url, apikey):
    try:
        u = urllib.parse.urlsplit(url)
        q = urllib.parse.parse_qs(u.query, keep_blank_values=True)
        q["apikey"] = [apikey]
        new_q = urllib.parse.urlencode(q, doseq=True)
        return urllib.parse.urlunsplit((u.scheme, u.netloc, u.path, new_q, u.fragment))
    except Exception:
        return url


def get_wagas_playlist_url():
    base = get_setting("base_url", DEFAULT_BASE).strip() or DEFAULT_BASE
    apikey = get_apikey()
    if not apikey:
        return ""

    if "apikey=" in base:
        if base.endswith("apikey="):
            return base + urllib.parse.quote(apikey)
        return replace_apikey_in_url(base, apikey)

    sep = "&" if "?" in base else "?"
    return "%s%sapikey=%s" % (base, sep, urllib.parse.quote(apikey))


# ---------- Parsing ----------
def first_text(node, tags):
    for t in tags:
        el = node.find(t)
        if el is not None and (el.text or "").strip():
            return (el.text or "").strip()
    return ""


def parse_xml_items(text):
    items = []
    root = ET.fromstring(text)

    candidates = []
    for path in [".//item", ".//channel", ".//entry", ".//stream", ".//media", ".//video"]:
        candidates.extend(root.findall(path))

    if not candidates:
        candidates = list(root)

    for n in candidates:
        title = first_text(n, ["title", "name", "label"]) or (n.get("title") or n.get("name") or "").strip()
        url = first_text(n, ["url", "link", "stream", "m3u8", "src"]) or (n.get("url") or n.get("link") or "").strip()
        group = first_text(n, ["group", "category", "genre"]) or (n.get("group") or n.get("category") or "").strip()
        logo = first_text(n, ["logo", "image", "thumb", "icon", "poster"]) or (n.get("logo") or n.get("image") or "").strip()

        if not url:
            link = n.find("link")
            if link is not None:
                url = (link.get("href") or "").strip()

        if title and url:
            items.append({"title": title, "url": url, "group": group or "Alle", "logo": logo or ""})

    seen = set()
    uniq = []
    for it in items:
        key = (it["title"], it["url"])
        if key in seen:
            continue
        seen.add(key)
        uniq.append(it)
    return uniq


_ATTR_RE = re.compile(r'(\w[\w\-]*)="([^"]*)"')


def parse_m3u_items(text):
    items = []
    lines = [ln.strip() for ln in (text or "").splitlines() if ln.strip()]

    current = None
    for ln in lines:
        if ln.startswith("#EXTINF"):
            attrs = dict(_ATTR_RE.findall(ln))
            parts = ln.split(",", 1)
            name = (parts[1].strip() if len(parts) > 1 else "Stream").strip() or "Stream"
            current = {
                "title": name,
                "group": (attrs.get("group-title") or attrs.get("group") or "Alle").strip() or "Alle",
                "logo": (attrs.get("tvg-logo") or attrs.get("logo") or "").strip(),
            }
            continue
        if ln.startswith("#"):
            continue
        if current:
            current["url"] = ln
            items.append(current)
            current = None

    return items


def parse_items(text):
    stripped = (text or "").lstrip()
    if not stripped:
        return []
    if stripped.startswith("<"):
        return parse_xml_items(text)
    if "#EXTM3U" in stripped or stripped.startswith("#EXTINF"):
        return parse_m3u_items(text)
    try:
        return parse_xml_items(text)
    except Exception:
        return []


def group_items(items):
    groups = {}
    for it in items:
        g = (it.get("group") or "Alle").strip() or "Alle"
        groups.setdefault(g, []).append(it)
    return groups


def pretty_category(name):
    return CATEGORY_PRETTY.get(name, name)


def category_icon(name):
    return CATEGORY_ICON.get(name, "DefaultFolder.png")


def category_fanart(name):
    return CATEGORY_FANART.get(name, DEFAULT_FANART)


def source_icon(source):
    return SOURCE_ICON.get(source, "DefaultFolder.png")


def source_fanart(source):
    return SOURCE_FANART.get(source, DEFAULT_FANART)


# ---------- UI ----------
def add_folder(label, mode, thumb=None, fanart=None, **params):
    li = xbmcgui.ListItem(label=label)
    art = {}
    if thumb:
        art.update({"icon": thumb, "thumb": thumb, "poster": thumb})
    else:
        art.update({"icon": "DefaultFolder.png", "thumb": "DefaultFolder.png"})
    if fanart:
        art["fanart"] = fanart
    li.setArt(art)
    xbmcplugin.addDirectoryItem(HANDLE, build_plugin_url(mode=mode, **params), li, isFolder=True)


def add_action(label, mode, thumb=None, fanart=None):
    li = xbmcgui.ListItem(label=label)
    art = {"icon": thumb or "DefaultAddon.png", "thumb": thumb or "DefaultAddon.png"}
    if fanart:
        art["fanart"] = fanart
    li.setArt(art)
    xbmcplugin.addDirectoryItem(HANDLE, build_plugin_url(mode=mode), li, isFolder=False)


def list_root():
    xbmcplugin.setPluginCategory(HANDLE, "Wagasworld")
    xbmcplugin.setContent(HANDLE, "videos")

    add_folder("🌍 Wagasworld Streams", "list_source", thumb=source_icon("wagas"), fanart=source_fanart("wagas"), source="wagas")
    add_folder("📺 TVworld Streams", "list_source", thumb=source_icon("tvworld"), fanart=source_fanart("tvworld"), source="tvworld")
    add_folder("⭐ Favoriten", "fav_folders", thumb=source_icon("favs"), fanart=source_fanart("favs"))
    add_folder("🕒 Zuletzt gesehen", "list_recent", thumb=source_icon("recent"), fanart=source_fanart("recent"))
    add_action("🔑 API-Key setzen/ändern", "setkey")

    xbmcplugin.endOfDirectory(HANDLE, cacheToDisc=True)


def load_source_items(source):
    if source == "tvworld":
        text, _ = get_cached_text("tvworld", TVWORLD_URL)
        items = parse_m3u_items(text)
        for it in items:
            it["source"] = "tvworld"
        return items

    playlist_url = get_wagas_playlist_url()
    if not playlist_url:
        return None
    text, _ = get_cached_text("wagas", playlist_url)
    items = parse_items(text)
    for it in items:
        it["source"] = "wagas"
    return items


def list_source(source):
    xbmcplugin.setPluginCategory(HANDLE, "Wagasworld" if source == "wagas" else "TVworld")
    xbmcplugin.setContent(HANDLE, "videos")

    try:
        items = load_source_items(source)
    except Exception as e:
        log("Quelle laden fehlgeschlagen (%s): %s" % (source, e), xbmc.LOGERROR)
        notify("Wagasworld", "Quelle konnte nicht geladen werden.")
        xbmcplugin.endOfDirectory(HANDLE, cacheToDisc=False)
        return

    if items is None:
        notify("Wagasworld", "Kein API-Key gesetzt.")
        xbmcplugin.endOfDirectory(HANDLE, cacheToDisc=False)
        return

    if not items:
        notify("Wagasworld", "Keine Einträge gefunden.")
        xbmcplugin.endOfDirectory(HANDLE, cacheToDisc=False)
        return

    groups = group_items(items)
    ordered = []
    if "Alle" in groups:
        ordered.append(("Alle", groups["Alle"]))
    for k in sorted([g for g in groups.keys() if g != "Alle"], key=lambda s: s.lower()):
        ordered.append((k, groups[k]))

    for g, arr in ordered:
        add_folder("%s  (%d)" % (pretty_category(g), len(arr)), "list_group",
                   thumb=category_icon(g), fanart=category_fanart(g), source=source, group=g)

    xbmcplugin.endOfDirectory(HANDLE, cacheToDisc=True)


def list_group(source, group_name):
    xbmcplugin.setPluginCategory(HANDLE, pretty_category(group_name))
    xbmcplugin.setContent(HANDLE, "videos")

    try:
        items = load_source_items(source)
    except Exception as e:
        log("Quelle laden fehlgeschlagen (%s): %s" % (source, e), xbmc.LOGERROR)
        notify("Wagasworld", "Quelle konnte nicht geladen werden.")
        xbmcplugin.endOfDirectory(HANDLE, cacheToDisc=False)
        return

    if items is None:
        notify("Wagasworld", "Kein API-Key gesetzt.")
        xbmcplugin.endOfDirectory(HANDLE, cacheToDisc=False)
        return

    groups = group_items(items)
    group_items_list = groups.get(group_name, [])

    if not group_items_list:
        notify("Wagasworld", "Keine Einträge in dieser Kategorie.")
        xbmcplugin.endOfDirectory(HANDLE, cacheToDisc=False)
        return

    cat_fan = category_fanart(group_name)

    for it in group_items_list:
        title = it.get("title", "Stream")
        url = it.get("url", "")
        logo = (it.get("logo") or "").strip()

        li = xbmcgui.ListItem(label=title)
        li.setInfo("video", {"title": title})

        art = {"fanart": cat_fan}
        if logo:
            art.update({"thumb": logo, "icon": logo, "poster": logo})
        li.setArt(art)

        li.setProperty("IsPlayable", "true")

        cm = []
        if is_fav(url):
            cm.append(("⭐ Wagasworld-Favorit entfernen", 'RunPlugin(%s)' % build_plugin_url(mode="fav_remove", url=url)))
            cm.append(("✏️ Favorit umbenennen", 'RunPlugin(%s)' % build_plugin_url(mode="fav_rename", url=url)))
            cm.append(("📁 Favorit-Ordner ändern", 'RunPlugin(%s)' % build_plugin_url(mode="fav_move_folder", url=url)))
        else:
            cm.append(("⭐ Wagasworld-Favorit hinzufügen", 'RunPlugin(%s)' % build_plugin_url(mode="fav_add", title=title, url=url, logo=logo, group=group_name, source=source)))
        li.addContextMenuItems(cm, replaceItems=False)

        xbmcplugin.addDirectoryItem(
            HANDLE,
            build_plugin_url(mode="play", url=url, title=title, logo=logo, group=group_name, source=source),
            li,
            isFolder=False
        )

    xbmcplugin.endOfDirectory(HANDLE, cacheToDisc=True)


def fav_folders():
    xbmcplugin.setPluginCategory(HANDLE, "Favoriten")
    xbmcplugin.setContent(HANDLE, "videos")

    favs = load_favs()
    if not favs:
        notify("Wagasworld", "Noch keine Favoriten gesetzt.")
        xbmcplugin.endOfDirectory(HANDLE, cacheToDisc=False)
        return

    folders = {}
    for f in favs:
        folder = (f.get("folder") or "Meine Favoriten").strip()
        folders.setdefault(folder, []).append(f)

    for folder in sorted(folders.keys(), key=lambda s: s.lower()):
        add_folder("📁 %s  (%d)" % (folder, len(folders[folder])), "list_favs_folder",
                   thumb=source_icon("favs"), fanart=source_fanart("favs"), folder=folder)

    add_folder("🧩 Alle Favoriten", "list_favs_folder", thumb=source_icon("favs"), fanart=source_fanart("favs"), folder="__ALL__")
    add_folder("↕️ Favoriten sortieren", "fav_sort_menu", thumb=source_icon("favs"), fanart=source_fanart("favs"))

    xbmcplugin.endOfDirectory(HANDLE, cacheToDisc=True)


def fav_sort_menu():
    xbmcplugin.setPluginCategory(HANDLE, "Favoriten sortieren")
    xbmcplugin.setContent(HANDLE, "videos")

    add_action("A‑Z (Ordner, dann Titel)", "fav_sort_alpha", thumb=source_icon("favs"), fanart=source_fanart("favs"))
    add_action("Zuletzt gespielt", "fav_sort_recent", thumb=source_icon("recent"), fanart=source_fanart("recent"))
    add_action("Zuletzt hinzugefügt", "fav_sort_added", thumb=source_icon("favs"), fanart=source_fanart("favs"))

    xbmcplugin.endOfDirectory(HANDLE, cacheToDisc=True)


def list_favs_folder(folder):
    xbmcplugin.setPluginCategory(HANDLE, "Favoriten")
    xbmcplugin.setContent(HANDLE, "videos")

    favs = load_favs()
    if not favs:
        notify("Wagasworld", "Noch keine Favoriten gesetzt.")
        xbmcplugin.endOfDirectory(HANDLE, cacheToDisc=False)
        return

    if folder != "__ALL__":
        favs = [f for f in favs if (f.get("folder") or "Meine Favoriten").strip() == folder]

    if not favs:
        notify("Wagasworld", "Keine Favoriten in diesem Ordner.")
        xbmcplugin.endOfDirectory(HANDLE, cacheToDisc=False)
        return

    fan = source_fanart("favs")

    for f in favs:
        title = f.get("title") or "Stream"
        url = f.get("url") or ""
        logo = (f.get("logo") or "").strip()

        li = xbmcgui.ListItem(label=title)
        li.setInfo("video", {"title": title})

        art = {"fanart": fan}
        if logo:
            art.update({"thumb": logo, "icon": logo, "poster": logo})
        else:
            art.update({"thumb": source_icon("favs"), "icon": source_icon("favs")})
        li.setArt(art)

        li.setProperty("IsPlayable", "true")

        cm = [
            ("⭐ Wagasworld-Favorit entfernen", 'RunPlugin(%s)' % build_plugin_url(mode="fav_remove", url=url)),
            ("✏️ Favorit umbenennen", 'RunPlugin(%s)' % build_plugin_url(mode="fav_rename", url=url)),
            ("📁 Favorit-Ordner ändern", 'RunPlugin(%s)' % build_plugin_url(mode="fav_move_folder", url=url)),
            ("⬆️ Favorit nach oben", 'RunPlugin(%s)' % build_plugin_url(mode="fav_move_up", url=url)),
            ("⬇️ Favorit nach unten", 'RunPlugin(%s)' % build_plugin_url(mode="fav_move_down", url=url)),
        ]
        li.addContextMenuItems(cm, replaceItems=False)

        xbmcplugin.addDirectoryItem(
            HANDLE,
            build_plugin_url(mode="play", url=url, title=title, logo=logo, group=f.get("group", ""), source=f.get("source", "wagas")),
            li,
            isFolder=False
        )

    xbmcplugin.endOfDirectory(HANDLE, cacheToDisc=True)


def list_recent():
    xbmcplugin.setPluginCategory(HANDLE, "Zuletzt gesehen")
    xbmcplugin.setContent(HANDLE, "videos")

    lst = load_recent()
    if not lst:
        notify("Wagasworld", "Noch kein Verlauf.")
        xbmcplugin.endOfDirectory(HANDLE, cacheToDisc=False)
        return

    fan = source_fanart("recent")

    for e in lst:
        title = e.get("title") or "Stream"
        url = e.get("url") or ""
        logo = (e.get("logo") or "").strip()
        src = e.get("source") or "wagas"
        grp = e.get("group") or "Alle"

        li = xbmcgui.ListItem(label=title)
        li.setInfo("video", {"title": title})

        art = {"fanart": fan}
        if logo:
            art.update({"thumb": logo, "icon": logo, "poster": logo})
        else:
            art.update({"thumb": source_icon("recent"), "icon": source_icon("recent")})
        li.setArt(art)

        li.setProperty("IsPlayable", "true")

        xbmcplugin.addDirectoryItem(
            HANDLE,
            build_plugin_url(mode="play", url=url, title=title, logo=logo, group=grp, source=src),
            li,
            isFolder=False
        )

    xbmcplugin.endOfDirectory(HANDLE, cacheToDisc=True)


def play_stream(stream_url, title="", logo="", group_name="Alle", source="wagas"):
    if not stream_url:
        notify("Wagasworld", "Ungültige Stream-URL.")
        return

    add_recent({"title": title or "Stream", "url": stream_url, "logo": logo or "", "group": group_name or "Alle", "source": source or "wagas"})
    update_fav_last_played(stream_url)

    li = xbmcgui.ListItem(path=stream_url)
    li.setProperty("IsPlayable", "true")
    xbmcplugin.setResolvedUrl(HANDLE, True, li)


def router(paramstring):
    params = dict(urllib.parse.parse_qsl(paramstring.lstrip("?")))
    mode = params.get("mode", "root")

    if mode == "play":
        return play_stream(params.get("url", ""), params.get("title", ""), params.get("logo", ""), params.get("group", "Alle"), params.get("source", "wagas"))

    if mode == "setkey":
        apikey = ask_for_apikey()
        notify("Wagasworld", "API-Key gespeichert." if apikey else "Kein API-Key gespeichert.")
        return list_root()

    if mode == "list_source":
        return list_source(params.get("source", "wagas"))

    if mode == "list_group":
        return list_group(params.get("source", "wagas"), params.get("group", "Alle"))

    if mode == "fav_folders":
        return fav_folders()

    if mode == "fav_sort_menu":
        return fav_sort_menu()

    if mode == "list_favs_folder":
        return list_favs_folder(params.get("folder", "__ALL__"))

    if mode == "fav_sort_alpha":
        fav_sort("alpha")
        notify("Wagasworld", "Favoriten sortiert (A‑Z).")
        return fav_folders()

    if mode == "fav_sort_recent":
        fav_sort("recent")
        notify("Wagasworld", "Favoriten sortiert (zuletzt gespielt).")
        return fav_folders()

    if mode == "fav_sort_added":
        fav_sort("added")
        notify("Wagasworld", "Favoriten sortiert (zuletzt hinzugefügt).")
        return fav_folders()

    if mode == "fav_add":
        folder = pick_folder()
        if not folder:
            notify("Wagasworld", "Abgebrochen.")
            return list_root()
        it = {
            "title": params.get("title", "Stream"),
            "url": params.get("url", ""),
            "logo": params.get("logo", ""),
            "group": params.get("group", "Alle"),
            "source": params.get("source", "wagas")
        }
        ok = add_fav(it, folder=folder)
        notify("Wagasworld", "Zu Favoriten hinzugefügt." if ok else "Konnte Favorit nicht speichern.")
        return list_root()

    if mode == "fav_remove":
        ok = remove_fav(params.get("url", ""))
        notify("Wagasworld", "Aus Favoriten entfernt." if ok else "Konnte Favorit nicht entfernen.")
        return list_root()

    if mode == "fav_rename":
        ok = fav_rename(params.get("url", ""))
        notify("Wagasworld", "Umbenannt." if ok else "Nicht geändert.")
        return list_root()

    if mode == "fav_move_folder":
        ok = fav_move_folder(params.get("url", ""))
        notify("Wagasworld", "Ordner geändert." if ok else "Nicht geändert.")
        return list_root()

    if mode == "fav_move_up":
        ok = fav_move(params.get("url", ""), -1)
        notify("Wagasworld", "Verschoben." if ok else "Nicht verschoben.")
        return fav_folders()

    if mode == "fav_move_down":
        ok = fav_move(params.get("url", ""), +1)
        notify("Wagasworld", "Verschoben." if ok else "Nicht verschoben.")
        return fav_folders()

    if mode == "list_recent":
        return list_recent()

    return list_root()


if __name__ == "__main__":
    try:
        router(sys.argv[2] if len(sys.argv) > 2 else "")
    except socket.timeout:
        notify("Wagasworld", "Timeout beim Laden.")
    except Exception as e:
        log("Unerwarteter Fehler: %s" % e, xbmc.LOGERROR)
        notify("Wagasworld", "Fehler – siehe Kodi Log.")
